Object.extend = function (destination, source) {
    for (var property in source)
        destination[property] = source[property];
    return destination;
};

var specialFunctionHelper = {
    // key word : shellHelper
    show : function (objectType) {
        var args = objectType.split(/\s+/);
        objectType = args[0];
        args = args.splice(1);

        if (objectType == "dbs" || objectType == "databases") {
            var dbs = db.adminCommand({ listDatabases: 1 });
            var dbinfo = [];
            var maxNameLength = 0;
            var maxGbDigits = 0;

            dbs.databases.forEach(function (x) {
                var sizeStr = (x.sizeOnDisk / 1024 / 1024 / 1024).toFixed(3);
                var nameLength = x.name.length;
                var gbDigits = sizeStr.indexOf(".");

                if (nameLength > maxNameLength)
                    maxNameLength = nameLength;
                if (gbDigits > maxGbDigits)
                    maxGbDigits = gbDigits;

                dbinfo.push({
                    name: x.name,
                    size: x.sizeOnDisk,
                    size_str: sizeStr,
                    name_size: nameLength,
                    gb_digits: gbDigits
                });
            });
            dbinfo.sort(compareOn('name'));

            dbinfo.forEach(function (db) {
                var namePadding = maxNameLength - db.name_size;
                var sizePadding = maxGbDigits - db.gb_digits;
                var padding = Array(namePadding + sizePadding + 3).join(" ");
                if (db.size > 1) {
                    print(db.name + padding + db.size_str + "GB");
                } else {
                    print(db.name + padding + "(empty)");
                }
            });

            return;
        } else if (objectType == "collections" || objectType == "tables") {
            db.getCollectionNames().forEach(function (collection) {
                print(collection);
            });
            return;
        } else if (objectType == "users") {
            db.getUsers().forEach(function (user) {
                print(tojson(user));
            });
            return;
        } else if (objectType == "roles") {
            db.getRoles({ showBuiltinRoles: true }).forEach(function (role) {
                print(tojson(role));
            });
            return;
        } else if (objectType == "profile") {
            if (db.system.profile.count() == 0) {
                print("db.system.profile is empty");
                print("Use db.setProfilingLevel(2) will enable profiling");
                print("Use db.system.profile.find() to show raw profile entries");
            } else {
                db.system.profile.find({ millis: { $gt: 0 } })
                    .sort({ $natural: -1 })
                    .limit(5)
                    .forEach(function (doc) {
                        print("" + doc.op + "\t" + doc.ns + " " + doc.millis + "ms " + String(doc.ts).substring(0, 24));
                        var resultStr = "";
                        for (var property in doc) {
                            if (property == "op" || property == "ns" || property == "millis" || property == "ts" || property == "tojson")
                                continue;

                            var val = doc[property];
                            var valType = typeof (val);

                            if (valType == "string" || valType == "number")
                                resultStr += property + ":" + val + " ";
                            else if (valType == "object")
                                resultStr += property + ":" + tojson(val) + " ";
                            else if (valType == "boolean")
                                resultStr += property + " ";
                            else
                                resultStr += property + ":" + val + " ";
                        }
                        print(resultStr + "\n\n");
                    });
            }
            return;
        } else if (objectType == "log") {
            var logName = "global";
            if (args.length > 0)
                logName = args[0];

            var result = db.adminCommand({ getLog: logName });
            if (!result.ok) {
                print("Error while trying to show " + logName + " log: " + result.errmsg);
                return;
            }

            for (var i = 0; i < result.log.length; i++) {
                print(result.log[i]);
            }
            return;
        } else if (objectType == "logs") {
            var result = db.adminCommand({ getLog: "*" });
            if (!result.ok) {
                print("Error while trying to show logs: " + result.errmsg);
                return;
            }

            for (var i = 0; i < result.names.length; i++) {
                print(result.names[i]);
            }
            return;
        } else if (objectType == "startupWarnings") {
            var dbDeclared, ex;
            try {
                dbDeclared = !!db;
            } catch (ex) {
                dbDeclared = false;
            }
            if (dbDeclared) {
                var result = db.adminCommand({ getLog: "startupWarnings" });
                if (result.ok) {
                    if (result.log.length == 0)
                        return;

                    print("Server has startup warnings: ");
                    for (var i = 0; i < result.log.length; i++) {
                        print(result.log[i]);
                    }
                    return;
                } else if (result.errmsg == "no such cmd: getLog") {
                    return;
                } else if (result.code == 13 || result.errmsg == "unauthorized" ||
                           result.errmsg == "need to login") {
                    return;
                } else {
                    print("Error while trying to show server startup warnings: " + result.errmsg);
                    return;
                }
            } else {
                print("Cannot show startupWarnings, \"db\" is not set");
                return;
            }
        } else if (objectType == "automationNotices") {
            var dbDeclared, ex;
            try {
                dbDeclared = !!db;
            } catch (ex) {
                dbDeclared = false;
            }

            if (dbDeclared) {
                var result = db.runCommand({ isMaster: 1, forShell: 1 });
                if (!result.ok) {
                    print("Note: Cannot determine if automation is active");
                    return;
                }

                if (result.hasOwnProperty("automationServiceDescriptor")) {
                    print("Note: This server is managed by automation service '" + result.automationServiceDescriptor + "'.");
                    print("Note: Many administrative actions are inappropriate, and may be automatically reverted.");
                }
                return;

            } else {
                print("Cannot show automationNotices, \"db\" is not set");
                return;
            }
        } else if (objectType == "freeMonitoring") {
            var dbDeclared, ex;
            try {
                dbDeclared = !!db;
            } catch (ex) {
                dbDeclared = false;
            }

            if (dbDeclared) {
                const freemonStatus = db.adminCommand({ getFreeMonitoringStatus: 1 });

                if (freemonStatus.ok) {
                    if (freemonStatus.state == 'enabled' &&
                        freemonStatus.hasOwnProperty('userReminder')) {
                        print("---");
                        print(freemonStatus.userReminder);
                        print("---");
                    } else if (freemonStatus.state === 'undecided') {
                        print(
                            "---\n" +
                            "Enable MongoDB's free cloud-based monitoring service to collect and display\n" +
                            "metrics about your deployment (disk utilization, CPU, operation statistics,\n" +
                            "etc).\n" + "\n" +
                            "The monitoring data will be available on a MongoDB website with a unique\n" +
                            "URL created for you. Anyone you share the URL with will also be able to\n" +
                            "view this page. MongoDB may use this information to make product\n" +
                            "improvements and to suggest MongoDB products and deployment options to you.\n" +
                            "\n" + "To enable free monitoring, run the following command:\n" +
                            "db.enableFreeMonitoring()\n" + "---\n");
                    }
                }

                return;
            } else {
                print("Cannot show freeMonitoring, \"db\" is not set");
                return;
            }
        }

        nav_throwError("show " + objectType + " is not supported");
    },
    use : function (dbName) {
        db = db.getSiblingDB(dbName);
        print("switched to db " + db.getName());
    }
};

var nav_parse = function (inputStr) {
    var parsedResult = Reflect.parse(inputStr);
    //return parsedResult;
    var parsedStrs = [inputStr];
    if (parsedResult.body) {
        if (parsedResult.body.length > 1) {
            parsedStrs = [];
            var lines = inputStr.split('\n');

            // use last end.column as startColumn instead of using start.column
            // because start.column is not really the starting position. 
            // e.g. 'function a(){};' start.position is 9 instead of 0
            var previousColumn = 0;
            var previousLine = 1;
            for (i = 0; i < parsedResult.body.length; i++) {
                var start = parsedResult.body[i].loc.start;
                var end = parsedResult.body[i].loc.end;
                var parsedStr = '';
                var startLine = start.line;
                if (startLine != previousLine)
                    previousColumn = 0;
                //previousLine = start.line;
                //previousColumn = start.column;
                while (startLine < end.line) {
                    parsedStr += lines[startLine - 1].substr(previousColumn, lines[startLine - 1].length - previousColumn) + '\n';
                    startLine++;
                    previousColumn = 0;
                }
                parsedStr += lines[startLine - 1].substr(previousColumn, end.column - previousColumn);
                previousLine = end.line;
                previousColumn = end.column;
                
                //if (parsedResult.body[i].type == 'FunctionDeclaration') {
                //    parsedStrs.push('function ' + parsedStr + '}');
                //}
                //else
                    if (parsedResult.body[i].type != 'EmptyStatement')
                    parsedStrs.push(parsedStr);
            }
        }
    }
    return parsedStrs;
}

var hex_md5 = function (text) {
    if (!isString(text))
        nav_throwError("hex_md5 should have a string argument");
    return this.forwardToCustomFunction("globalHexMD5", text);
};

var convertShardKeyToHashed = function (obj) {
    var pipeline = [
        { $limit: 1 },
        { $project: { _id: { $toHashedIndexKey: { $literal: obj } } } },
    ];
    var result;
    for (const approach of [
        () =>
            db.getSiblingDB('_NAV_FakeDb').aggregate([
                { $documents: [{}] }, ...pipeline
            ]),
        () => db.getSiblingDB('admin').aggregate([
            { $documents: [{}] }, ...pipeline
        ]),
        () =>
            db.getSiblingDB('admin').getCollection('system.version').aggregate(pipeline),
        () =>
            db.getSiblingDB('local')
                .getCollection('oplog.rs')
                .aggregate([
                    { $collStats: {} }, ...pipeline
                ])
    ]) {
        result = approach().next();
        if (result) break;
    }
    if (!result) {
        nav_throwError('fail to run convertShardKeyToHashed() -- tried $documents and aggregating on admin.system.version and local.oplog.rs');
    }
    return result._id;
};

friendlyEqual = function (a, b) {
    if (a == b)
        return true;

    a = tojson(a, false, true);
    b = tojson(b, false, true);

    if (a == b)
        return true;

    var clean = function (s) {
        s = s.replace(/NumberInt\((\-?\d+)\)/g, "$1");
        return s;
    };

    a = clean(a);
    b = clean(b);

    if (a == b)
        return true;

    return false;
};

var indentStr = function (indent, str) {
    if (isUndefined(str)) {
        str = indent;
        indent = 0;
    }
    if (indent > 0) {
        indent = (new Array(indent + 1)).join(" ");
        str = indent + str.replace(/\n/g, "\n" + indent);
    }
    return str;
};

tojsononeline = function (x) {
    return tojson(x, " ", true);
};

tojson = function (obj, indent, nolint, depth) {
    if (obj === null)
        return "null";

    if (isUndefined(obj))
        return "undefined";

    if (!indent)
        indent = "";

    if (!isNumber(depth))
        depth = 0;

    switch (typeof obj) {
        case "string": {
            var array = new Array(obj.length + 1);
            array[0] = '"';
            for (var i = 0; i < obj.length; i++) {
                switch (obj[i]) {
                    case '"':
                        array[array.length] = '\\"';
                        break;
                    case '\\':
                        array[array.length] = '\\\\';
                        break;
                    case '\b':
                        array[array.length] = '\\b';
                        break;
                    case '\f':
                        array[array.length] = '\\f';
                        break;
                    case '\n':
                        array[array.length] = '\\n';
                        break;
                    case '\r':
                        array[array.length] = '\\r';
                        break;
                    case '\t':
                        array[array.length] = '\\t';
                        break;

                    default: {
                        var code = obj.charCodeAt(i);
                        if (code < 0x20) {
                            array[array.length] =
                                (code < 0x10 ? '\\u000' : '\\u00') + code.toString(16);
                        } else {
                            array[array.length] = obj[i];
                        }
                    }
                }
            }

            return array.join('') + "\"";
        }
        case "number":
        case "boolean":
            return "" + obj;
        case "object": {
            var jsonStr = tojsonObject(obj, indent, nolint, depth);
            if ((nolint == null || nolint == true) && jsonStr.length < 80 &&
                (indent == null || indent.length == 0)) {
                jsonStr = jsonStr.replace(/[\t\r\n]+/gm, " ");
            }
            return jsonStr;
        }
        case "function":
            if (obj === MinKey || obj === MaxKey)
                return obj.tojson();
            return obj.toString();
        default:
            nav_throwError("tojson can't handle type " + (typeof obj));
    }

};
tojson.MAX_DEPTH = 100;

tojsonObject = function (obj, indent, nolint, depth) {
    if (!isNumber(depth))
        depth = 0;

    var lineEnding = nolint ? " " : "\n";
    var tabSpace = nolint ? "" : "\t";

    if (!indent)
        indent = "";

    if (isFunction(obj.tojson) && obj.tojson != tojson)
        return obj.tojson(indent, nolint, depth);

    if (obj.constructor && isFunction(obj.constructor.tojson) &&
        obj.constructor.tojson != tojson) {
        return obj.constructor.tojson(obj, indent, nolint, depth);
    }

    if (obj instanceof Error)
        return obj.toString();

    try {
        obj.toString();
    } catch (e) {
        return "[Object]";
    }

    if (depth > tojson.MAX_DEPTH)
        return "[Object]";

    var jsonStr = "{" + lineEnding;

    indent += tabSpace;

    var keys = obj;
    if (isFunction(obj._simpleKeys))
        keys = obj._simpleKeys();
    var fieldStrings = [];
    for (var key in keys) {
        if (!keys.hasOwnProperty(key))
            continue;

        var val = obj[key];

        if (!isUndefined(NAVDatabase) && val == NAVDatabase.prototype)
            continue;
        if (!isUndefined(NAVCollection) && val == NAVCollection.prototype)
            continue;

        fieldStrings.push(indent + "\"" + key + "\" : " + tojson(val, indent, nolint, depth + 1));
    }

    if (fieldStrings.length > 0)
        jsonStr += fieldStrings.join("," + lineEnding);
    else
        jsonStr += indent;
    jsonStr += lineEnding;

    indent = indent.substring(1);
    return jsonStr + indent + "}";
};

String.prototype.pad = function (length, isRight, char) {
    if (isUndefined(char))
        char = ' ';
    var resultStr = this;
    for (var i = length - resultStr.length; i > 0; i--) {
        if (isRight)
            resultStr = resultStr + char;
        else
            resultStr = char + resultStr;
    }
    return resultStr;
};

function HexToBase64(hex) {
    var base64Digits = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
    var base64 = "";
    var group;
    for (var i = 0; i < 30; i += 6) {
        group = parseInt(hex.substr(i, 6), 16);
        base64 += base64Digits[(group >> 18) & 0x3f];
        base64 += base64Digits[(group >> 12) & 0x3f];
        base64 += base64Digits[(group >> 6) & 0x3f];
        base64 += base64Digits[group & 0x3f];
    }
    group = parseInt(hex.substr(30, 2), 16);
    base64 += base64Digits[(group >> 2) & 0x3f];
    base64 += base64Digits[(group << 4) & 0x3f];
    base64 += "==";
    return base64;
}

isUndefined = function (value) {
    return typeof (value) == "undefined";
};

isFunction = function (value) {
    return typeof (value) == "function";
};

isString = function (value) {
    return typeof (value) == "string";
};

isNumber = function (value) {
    return typeof (value) == "number";
};

isObject = function (value) {
    return typeof (value) == "object";
};

isBoolean = function (value) {
    return typeof (value) == "boolean";
};

function nav_throwError(message, error) {
    var errorObj = new Error(message);
    if (!isUndefined(error)) {
        if (error.writeError)
            errorObj.code = error.writeError.code;
        else if (error.code)
            errorObj.code = error.code;
        else
            errorObj.code = error;
    }

    throw errorObj;
}

compare = function (firstObj, secondObj) {
    return (firstObj == secondObj ? 0 : (firstObj < secondObj ? -1 : 1));
};

compareOn = function (fieldName) {
    return function (firstObj, secondObj) {
        return compare(firstObj[fieldName], secondObj[fieldName]);
    };
};

print = function (obj) {
    var resultStr = "";
    var tmpArray = Array.from(arguments);
    for (var i = 0; i < tmpArray.length; i++) {
        if (isUndefined(tmpArray[i]) || tmpArray[i] == null)
            resultStr += "[unknown type]";
        else
            resultStr += tmpArray[i].toString();
        if (i != tmpArray.length-1)
            resultStr += " ";
    }

    this.forwardToCustomFunction("globalPrint", resultStr);
};

printjson = function (obj) {
    this.forwardToCustomFunction("globalPrintJson", tojson(obj));
};

printjsononeline = function (obj) {
    this.forwardToCustomFunction("globalPrintJsonOneLine", tojsononeline(obj));
};

convertToValidInt64String = function (obj) {
    this.forwardToCustomFunction("globalConvertToValidInt64String", obj);
};

convertToValidIntString = function (obj) {
    return this.forwardToCustomFunction("globalConvertToValidIntString", obj);
};

Binary.createFromBase64 = function (base64String, subType) {
    if (!isString(base64String))
        nav_throwError("Binary.createFromBase64 should have a string base64");

    return nav_binary_createFromBase64(base64String, subType);
};

Binary.createFromHexString = function (hexadecimalString) {
    if (!isString(hexadecimalString))
        nav_throwError("Binary.createFromHexString should have a string hexadecimal");

    return nav_binary_createFromHexString(hexadecimalString);
};

nav_binary_createFromBase64 = function (base64String, subType) {
    return this.forwardToCustomFunction("globalBinaryCreateFromBase64", base64String, subType);
};

nav_binary_createFromHexString = function (hexadecimalString) {
    return this.forwardToCustomFunction("globalBinaryCreateFromHexString", hexadecimalString);
};

ObjectId.createFromBase64 = function (base64String, subType) {
    if (!isString(base64String))
        nav_throwError("Binary.createFromBase64 should have a string base64");

    return nav_objectId_createFromBase64(base64String, subType);
};

ObjectId.createFromHexString = function (hexadecimalString) {
    if (!isString(hexadecimalString))
        nav_throwError("Binary.createFromHexString should have a string hexadecimal");

    return nav_objectId_createFromHexString(hexadecimalString);
};

nav_objectId_createFromBase64 = function (base64String, subType) {
    return this.forwardToCustomFunction("globalObjectIdCreateFromBase64", base64String, subType);
};

nav_objectId_createFromHexString = function (hexadecimalString) {
    return this.forwardToCustomFunction("globalObjectIdCreateFromHexString", hexadecimalString);
};

nav_convertToValidInt64String = function (input) {
    if (typeof input === 'undefined')
        return "0";
    if (typeof input === 'boolean')
        return input ? "1" : "0";
    // note: input.toString() will truncate JS Number precision, JS Numbers are IEEE-754 binary double-precision floats
    return input.toString();
}

nav_convertToValidInt = function (input) {
    if (typeof input === 'undefined')
        return "0";
    if (typeof input === 'boolean')
        return input ? 1 : 0;
    return parseInt(input);
}

sleep = function (milliseconds) {
    if (isUndefined(milliseconds) || !isNumber(milliseconds))
        nav_throwError("sleep takes a single numeric argument -- sleep(millisecondss)");
    this.forwardToCustomFunction("globalSleep", milliseconds);
}

function _getErrorWithCode(codeOrObj, message) {
    var errorObj = new Error(message);
    if (codeOrObj != undefined) {
        if (codeOrObj.writeError)
            errorObj.code = codeOrObj.writeError.code;
        else if (codeOrObj.code)
            errorObj.code = codeOrObj.code;
        else
            errorObj.code = codeOrObj;
    }

    return errorObj;
}

doassert = function(msg, obj) {
    if (isFunction(msg))
        msg = msg();

    if (isObject(msg))
        msg = tojson(msg);

    if (isString(msg) && msg.indexOf("assert") == 0)
        print(msg);
    else
        print("assert: " + msg);

    var errorObj;
    if (obj)
        errorObj = _getErrorWithCode(obj, msg);
    else
        errorObj = Error(msg);
    print(errorObj.stack);
    throw errorObj;
};

assert = function(needReturn, msg) {
    if (arguments.length > 2)
        doassert("Too many parameters to assert().");
    if (arguments.length > 1 && !isString(msg))
        doassert("Non-string 'msg' parameters are invalid for assert().");
    if (assert._debug && msg)
        print("in assert for: " + msg);
    if (needReturn)
        return;
    doassert(isUndefined(msg) ? "assert failed" : "assert failed : " + msg);
};

assert._debug = false;

assert.soon = function(func, msg, timeout, interval) {
    if (assert._debug && msg)
        print("in assert for: " + msg);

    if (msg) {
        if (!isFunction(msg))
            msg = "assert.soon failed, msg:" + msg;
    } else {
        msg = "assert.soon failed: " + func;
    }

    var start = new Date();
    timeout = timeout || 5 * 60 * 1000;
    interval = interval || 200;
    var last;
    while (true) {
        if (isString(func)) {
            if (eval(func))
                return;
        } else {
            if (func())
                return;
        }

        diff = (new Date()).getTime() - start.getTime();
        if (diff > timeout)
            doassert(msg);
        sleep(interval);
    }
};

